<?php
// Prevent direct access
if (!defined('ACCESS_ALLOWED')) {
    header('HTTP/1.0 403 Forbidden');
    exit;
}

/**
 * Generate a unique student ID
 */
function generateStudentId() {
    global $pdo;
    
    $prefix = 'STU';
    $year = date('Y');
    
    // Get the last student ID
    $sql = "SELECT student_id FROM students ORDER BY id DESC LIMIT 1";
    $stmt = $pdo->query($sql);
    $lastStudent = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($lastStudent) {
        $lastId = intval(substr($lastStudent['student_id'], -4));
        $newId = $lastId + 1;
    } else {
        $newId = 1;
    }
    
    return $prefix . $year . str_pad($newId, 4, '0', STR_PAD_LEFT);
}

/**
 * Get setting value by key
 */
function getSetting($key, $default = '') {
    global $pdo;
    
    $sql = "SELECT setting_value FROM settings WHERE setting_key = :key";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':key', $key);
    $stmt->execute();
    
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result ? $result['setting_value'] : $default;
}

/**
 * Format currency amount
 */
function formatCurrency($amount) {
    $symbol = getSetting('currency_symbol', '₦');
    return $symbol . number_format($amount, 2);
}

/**
 * Get student name by ID
 */
function getStudentName($id) {
    global $pdo;
    
    $sql = "SELECT first_name, last_name FROM students WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':id', $id);
    $stmt->execute();
    
    $student = $stmt->fetch(PDO::FETCH_ASSOC);
    return $student ? $student['first_name'] . ' ' . $student['last_name'] : 'Unknown Student';
}

/**
 * Get fee category name by ID
 */
function getFeeCategoryName($id) {
    global $pdo;
    
    $sql = "SELECT name FROM fee_categories WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':id', $id);
    $stmt->execute();
    
    $category = $stmt->fetch(PDO::FETCH_ASSOC);
    return $category ? $category['name'] : 'Unknown Category';
}

/**
 * Check if student has unpaid fees
 */
function hasUnpaidFees($student_id) {
    global $pdo;
    
    $sql = "SELECT COUNT(*) FROM fee_allocations 
            WHERE student_id = :student_id AND status IN ('pending', 'overdue')";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':student_id', $student_id);
    $stmt->execute();
    
    return $stmt->fetchColumn() > 0;
}

/**
 * Get total unpaid amount for a student
 */
function getTotalUnpaidAmount($student_id) {
    global $pdo;
    
    $sql = "SELECT SUM(amount) FROM fee_allocations 
            WHERE student_id = :student_id AND status IN ('pending', 'overdue')";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':student_id', $student_id);
    $stmt->execute();
    
    return $stmt->fetchColumn() ?: 0;
}

/**
 * Generate receipt number
 */
function generateReceiptNumber() {
    global $pdo;
    
    $prefix = getSetting('receipt_prefix', 'EXCEL');
    $year = date('Y');
    
    // Get the last receipt number
    $sql = "SELECT id FROM payments ORDER BY id DESC LIMIT 1";
    $stmt = $pdo->query($sql);
    $lastPayment = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($lastPayment) {
        $lastId = $lastPayment['id'];
        $newId = $lastId + 1;
    } else {
        $newId = 1;
    }
    
    return $prefix . $year . str_pad($newId, 5, '0', STR_PAD_LEFT);
}

/**
 * Send notification email
 */
function sendNotificationEmail($to, $subject, $message) {
    // This is a placeholder function - implement actual email sending
    // using PHPMailer, SendGrid, or your preferred email service
    $headers = "From: " . getSetting('school_email', 'noreply@example.com') . "\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    
    return mail($to, $subject, $message, $headers);
}

/**
 * Log activity
 */
function logActivity($user_id, $action, $details = '') {
    global $pdo;
    
    $sql = "INSERT INTO activity_log (user_id, action, details, ip_address, user_agent) 
            VALUES (:user_id, :action, :details, :ip_address, :user_agent)";
    
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':user_id', $user_id);
    $stmt->bindParam(':action', $action);
    $stmt->bindParam(':details', $details);
    $stmt->bindParam(':ip_address', $_SERVER['REMOTE_ADDR']);
    $stmt->bindParam(':user_agent', $_SERVER['HTTP_USER_AGENT']);
    
    return $stmt->execute();
}

/**
 * Get dashboard statistics
 */
function getDashboardStats() {
    global $pdo;
    
    $stats = [];
    
    // Total students
    $sql = "SELECT COUNT(*) FROM students";
    $stats['total_students'] = $pdo->query($sql)->fetchColumn();
    
    // Active students
    $sql = "SELECT COUNT(*) FROM students WHERE status = 'active'";
    $stats['active_students'] = $pdo->query($sql)->fetchColumn();
    
    // Total receivable
    $sql = "SELECT SUM(amount) FROM fee_allocations WHERE status IN ('pending', 'overdue')";
    $stats['total_receivable'] = $pdo->query($sql)->fetchColumn() ?: 0;
    
    // Today's collection
    $sql = "SELECT SUM(amount) FROM payments WHERE DATE(payment_date) = CURDATE()";
    $stats['today_collection'] = $pdo->query($sql)->fetchColumn() ?: 0;
    
    // Monthly collection
    $sql = "SELECT SUM(amount) FROM payments WHERE MONTH(payment_date) = MONTH(CURDATE()) 
            AND YEAR(payment_date) = YEAR(CURDATE())";
    $stats['monthly_collection'] = $pdo->query($sql)->fetchColumn() ?: 0;
    
    // Overdue fees count
    $sql = "SELECT COUNT(*) FROM fee_allocations WHERE status = 'overdue'";
    $stats['overdue_fees'] = $pdo->query($sql)->fetchColumn();
    
    return $stats;
}

/**
 * Check for overdue fees and update status
 */
function checkOverdueFees() {
    global $pdo;
    
    $sql = "UPDATE fee_allocations SET status = 'overdue' 
            WHERE due_date < CURDATE() AND status = 'pending'";
    
    $stmt = $pdo->prepare($sql);
    return $stmt->execute();
}

/**
 * Get recent activities
 */
function getRecentActivities($limit = 10) {
    global $pdo;
    
    $sql = "SELECT a.*, u.username 
            FROM activity_log a 
            JOIN users u ON a.user_id = u.id 
            ORDER BY a.created_at DESC 
            LIMIT :limit";
    
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Check for overdue fees on every page load (could be moved to a cron job)
checkOverdueFees();
?>